<?php

if (!defined('ABSPATH')) {
    die('Access Denied!');
}

// Ensure this logic runs only for logged-in admins
if (is_user_logged_in() && current_user_can('administrator') && is_admin()) {
    global $wpdb;
    $charset = $wpdb->get_charset_collate();

    // Define table names
    $table_name = $wpdb->prefix . 'opeform_mailbox';
    $table_name_terms = $wpdb->prefix . 'opeform_mailbox_terms';

    // Check and create the 'opeform_mailbox' table if it doesn't exist
    if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name)) != $table_name) {
        $table_sql = "CREATE TABLE IF NOT EXISTS $table_name (
                mail_id BIGINT(20) NOT NULL AUTO_INCREMENT,
                from_name VARCHAR(50),
                from_email VARCHAR(255),
                mail_category BIGINT(20),
                mail_data LONGTEXT,
                http_data LONGTEXT,
                attachment_images LONGTEXT,
                attachment_files LONGTEXT,
                created TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY mail_id(mail_id),
                INDEX idx_from_email (from_email),  -- Index on from_email for faster lookups
                INDEX idx_mail_category (mail_category),  -- Index on mail_category for filtering
                INDEX idx_created (created)  -- Index on created for sorting by creation date
            ) $charset;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($table_sql);
    }

    // Check and create the 'opeform_mailbox_terms' table if it doesn't exist
    if ($wpdb->get_var($wpdb->prepare("SHOW TABLES LIKE %s", $table_name_terms)) != $table_name_terms) {
        $table_sql_terms = "CREATE TABLE IF NOT EXISTS $table_name_terms (
                term_id BIGINT(20) NOT NULL AUTO_INCREMENT,
                term_name VARCHAR(100),
                term_slug VARCHAR(100),
                http_data LONGTEXT,
                created TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP,
                UNIQUE KEY term_id(term_id),
                INDEX idx_term_name (term_name),  -- Index on term_name for searching
                INDEX idx_term_slug (term_slug)  -- Index on term_slug for searching
            ) $charset;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($table_sql_terms);
    }

    // Flush rewrite rules only after table creation or modification
    flush_rewrite_rules();
}



class opeSaveMailData
{
    public static function createSlug($str, $delimiter = '-')
    {
        $slug = $str;
        if (preg_match('/[\x{4e00}-\x{9fa5}]+/u', $str)) {
            $str = preg_replace('/\s+/', '_', $str);
            $slug = $str;
        } else {
            $slug = strtolower(trim(preg_replace('/[\s-]+/', $delimiter, preg_replace('/[^A-Za-z0-9-]+/', $delimiter, preg_replace('/[&]/', 'and', preg_replace('/[\']/', '', iconv('UTF-8', 'ASCII//TRANSLIT', $str))))), $delimiter));
        }
        return $slug;
    }

    public static function insert_post($post_data)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'opeform_mailbox';

        $from_name = sanitize_text_field($post_data['user_client_name']);
        $from_email = sanitize_email($post_data['user_client_email']);
        $mail_category = sanitize_text_field($post_data['category']);
        $mail_data = maybe_serialize($post_data['user_email_data']);
        $http_data = maybe_serialize($post_data['user_http_data']);
        $attachment_images = maybe_serialize($post_data['user_attachment_images']);
        $attachment_files = maybe_serialize($post_data['user_attachment_files']);

        $data = array(
            'from_name' => $from_name,
            'from_email' => $from_email,
            'mail_category' => $mail_category,
            'mail_data' => $mail_data,
            'http_data' => $http_data,
            'attachment_images' => $attachment_images,
            'attachment_files' => $attachment_files,
        );

        $inserted = $wpdb->insert($table_name, $data);
        if ($inserted === false) {
            error_log('Failed to insert post data into opeform_mailbox table.');
            return false;
        }

        return $wpdb->insert_id;
    }

    public static function insert_term($term, $http_data)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'opeform_mailbox_terms';
        $slug = self::createSlug($term);

        $term = sanitize_text_field($term);
        $http_data = maybe_serialize($http_data);

        $data = array(
            'term_name' => $term,
            'term_slug' => $slug,
            'http_data' => $http_data,
        );

        $inserted = $wpdb->insert($table_name, $data);
        if ($inserted === false) {
            error_log('Failed to insert term into opeform_mailbox_terms table.');
            return false;
        }

        return array('term_id' => $wpdb->insert_id);
    }

    public static function term_exists($term)
    {
        global $wpdb;
        $table_name = $wpdb->prefix . 'opeform_mailbox_terms';
        $slug = self::createSlug($term);

        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT * FROM $table_name WHERE term_slug = %s",
                $slug
            )
        );

        if (empty($result)) {
            return false;
        } else {
            return array('term_id' => $result->term_id);
        }
    }
}