<?php
/*
Plugin Name: Ten Minutes Custom Post Type Generator
Description: Ten Minutes Custom Post Type Generator lets you create Custom Post Types and custom Taxonomies in easy way.
Version: 18.02.02
Author: 10 Minutes
Author URI: https://10.1pxeye.com/
*/
if ( ! function_exists( 'add_filter' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit();
}
class tmcptg 
{
	private $dir;
	private $path;
	private $version;
	function __construct() 
	{
		$this->dir = plugins_url( '', __FILE__ );
		$this->path = plugin_dir_path( __FILE__ );
		$this->version = '18.02.02';
		add_action( 'init', array($this, 'init') );
		add_action( 'init', array($this, 'tmcptg_create_custom_post_types') );
		add_action( 'admin_menu', array($this, 'tmcptg_admin_menu') );
		add_action( 'admin_enqueue_scripts', array($this, 'tmcptg_styles') );
		add_action( 'add_meta_boxes', array($this, 'tmcptg_create_meta_boxes') );
		add_action( 'save_post', array($this, 'tmcptg_save_post') );
		add_action( 'admin_init', array($this, 'tmcptg_plugin_settings_flush_rewrite') );
		add_action( 'manage_posts_custom_column', array($this, 'tmcptg_custom_columns'), 10, 2 );
		add_action( 'manage_posts_custom_column', array($this, 'tmcptg_tax_custom_columns'), 10, 2 );
		add_action( 'wp_prepare_attachment_for_js', array($this, 'wp_prepare_attachment_for_js'), 10, 3 );
		add_filter( 'manage_tmcptg_posts_columns', array($this, 'tmcptg_change_columns') );
		add_filter( 'manage_edit-tmcptg_sortable_columns', array($this, 'tmcptg_sortable_columns') );
		add_filter( 'manage_tmcptg_tax_posts_columns', array($this, 'tmcptg_tax_change_columns') );
		add_filter( 'manage_edit-tmcptg_tax_sortable_columns', array($this, 'tmcptg_tax_sortable_columns') );
		add_filter( 'post_updated_messages', array($this, 'tmcptg_post_updated_messages') );
		register_deactivation_hook( __FILE__, 'flush_rewrite_rules' );
		register_activation_hook( __FILE__, array($this, 'tmcptg_plugin_activate_flush_rewrite') );
	}
	public function init() 
	{
		$labels = array(
			'name' => __( 'Custom Post Type', 'ten-minutes-custom-post-type-generator' ),
			'singular_name' => __( 'Custom Post Type', 'ten-minutes-custom-post-type-generator' ),
			'add_new' => __( 'Add New' , 'ten-minutes-custom-post-type-generator' ),
			'add_new_item' => __( 'Add New Custom Post Type' , 'ten-minutes-custom-post-type-generator' ),
			'edit_item' =>  __( 'Edit Custom Post Type' , 'ten-minutes-custom-post-type-generator' ),
			'new_item' => __( 'New Custom Post Type' , 'ten-minutes-custom-post-type-generator' ),
			'view_item' => __('View Custom Post Type', 'ten-minutes-custom-post-type-generator' ),
			'search_items' => __('Search Custom Post Types', 'ten-minutes-custom-post-type-generator' ),
			'not_found' =>  __('No Custom Post Types found', 'ten-minutes-custom-post-type-generator' ),
			'not_found_in_trash' => __('No Custom Post Types found in Trash', 'ten-minutes-custom-post-type-generator' ),
		);
		register_post_type( 'tmcptg', array(
			'labels' => $labels,
			'public' => false,
			'show_ui' => true,
			'_builtin' =>  false,
			'capability_type' => 'page',
			'hierarchical' => false,
			'rewrite' => false,
			'query_var' => 'tmcptg',
			'supports' => array(
				'title'
			),
			'show_in_menu' => false,
		));
		$labels = array(
			'name' => __( 'Custom Taxonomies', 'ten-minutes-custom-post-type-generator' ),
			'singular_name' => __( 'Custom Taxonomy', 'ten-minutes-custom-post-type-generator' ),
			'add_new' => __( 'Add New' , 'ten-minutes-custom-post-type-generator' ),
			'add_new_item' => __( 'Add New Custom Taxonomy' , 'ten-minutes-custom-post-type-generator' ),
			'edit_item' =>  __( 'Edit Custom Taxonomy' , 'ten-minutes-custom-post-type-generator' ),
			'new_item' => __( 'New Custom Taxonomy' , 'ten-minutes-custom-post-type-generator' ),
			'view_item' => __('View Custom Taxonomy', 'ten-minutes-custom-post-type-generator' ),
			'search_items' => __('Search Custom Taxonomies', 'ten-minutes-custom-post-type-generator' ),
			'not_found' =>  __('No Custom Taxonomies found', 'ten-minutes-custom-post-type-generator' ),
			'not_found_in_trash' => __('No Custom Taxonomies found in Trash', 'ten-minutes-custom-post-type-generator' ),
		);
		register_post_type( 'tmcptg_tax', array(
			'labels' => $labels,
			'public' => false,
			'show_ui' => true,
			'_builtin' =>  false,
			'capability_type' => 'page',
			'hierarchical' => false,
			'rewrite' => false,
			'query_var' => 'tmcptg_tax',
			'supports' => array(
				'title'
			),
			'show_in_menu' => false,
		));
		if ( function_exists( 'add_image_size' ) ) {
			add_image_size( 'tmcptg_icon', 16, 16, true );
		}
	}
	public function tmcptg_admin_menu() 
	{
		add_menu_page( __('Post Types', 'ten-minutes-custom-post-type-generator' ), __('Post Types', 'ten-minutes-custom-post-type-generator' ), 'manage_options', 'edit.php?post_type=tmcptg', '', 'data:image/svg+xml;base64,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' );
		add_submenu_page( 'edit.php?post_type=tmcptg', __('Taxonomies', 'ten-minutes-custom-post-type-generator' ), __('Taxonomies', 'ten-minutes-custom-post-type-generator' ), 'manage_options', 'edit.php?post_type=tmcptg_tax' );
	}
	
	public function tmcptg_styles( $hook ) 
	{
        wp_register_style( 'tmcptg_add_edit_styles', $this->dir . '/assets/style.css' );
			wp_enqueue_style( 'tmcptg_add_edit_styles' );
		// register add / edit style
		if ( ( $hook == 'post-new.php' && isset($_GET['post_type']) && $_GET['post_type'] == 'tmcptg' ) || ( $hook == 'post.php' && isset($_GET['post']) && get_post_type( $_GET['post'] ) == 'tmcptg' ) || ( $hook == 'post-new.php' && isset($_GET['post_type']) && $_GET['post_type'] == 'tmcptg_tax' ) || ( $hook == 'post.php' && isset($_GET['post']) && get_post_type( $_GET['post'] ) == 'tmcptg_tax' ) ) {	
			wp_register_script( 'tmcptg_admin__add_edit_js', $this->dir . '/assets/script.js', 'jquery', '18.02.02', true );
			wp_enqueue_script( 'tmcptg_admin__add_edit_js' );
			wp_enqueue_media();
		}
	}
	
	public function tmcptg_create_custom_post_types()
	{
		$tmcptgs = array();
		$tmcptg_taxs = array();
		$get_tmcptg = array(
			'numberposts' 	   => -1,
			'post_type' 	   => 'tmcptg',
			'post_status'      => 'publish',
			'suppress_filters' => false,
		);
		$tmcptg_post_types = get_posts( $get_tmcptg );
		if ( $tmcptg_post_types ) {
			foreach ( $tmcptg_post_types as $tmcptg ) {
				$tmcptg_meta = get_post_meta( $tmcptg->ID, '', true );
				$tmcptg_name                = ( array_key_exists( 'tmcptg_name', $tmcptg_meta ) && $tmcptg_meta['tmcptg_name'][0] ? esc_html( $tmcptg_meta['tmcptg_name'][0] ) : 'no_name' );
				$tmcptg_label               = ( array_key_exists( 'tmcptg_label', $tmcptg_meta ) && $tmcptg_meta['tmcptg_label'][0] ? esc_html( $tmcptg_meta['tmcptg_label'][0] ) : $tmcptg_name );
				$tmcptg_singular_name       = ( array_key_exists( 'tmcptg_singular_name', $tmcptg_meta ) && $tmcptg_meta['tmcptg_singular_name'][0] ? esc_html( $tmcptg_meta['tmcptg_singular_name'][0] ) : $tmcptg_label );
				$tmcptg_description         = ( array_key_exists( 'tmcptg_description', $tmcptg_meta ) && $tmcptg_meta['tmcptg_description'][0] ? $tmcptg_meta['tmcptg_description'][0] : '' );
				$tmcptg_icon_url = ( array_key_exists( 'tmcptg_icon_url', $tmcptg_meta ) && $tmcptg_meta['tmcptg_icon_url'][0] ? $tmcptg_meta['tmcptg_icon_url'][0] : false );				
				
				$tmcptg_icon_slug = ( array_key_exists( 'tmcptg_icon_slug', $tmcptg_meta ) && $tmcptg_meta['tmcptg_icon_slug'][0] ? $tmcptg_meta['tmcptg_icon_slug'][0] : false );
				
				if ( !empty($tmcptg_icon_slug) ) {
					$tmcptg_icon_name = $tmcptg_icon_slug;
				} else {
					$tmcptg_icon_name = $tmcptg_icon_url;
				}
				
				$tmcptg_custom_rewrite_slug = ( array_key_exists( 'tmcptg_custom_rewrite_slug', $tmcptg_meta ) && $tmcptg_meta['tmcptg_custom_rewrite_slug'][0] ? esc_html( $tmcptg_meta['tmcptg_custom_rewrite_slug'][0] ) : $tmcptg_name );
				$tmcptg_menu_position       = ( array_key_exists( 'tmcptg_menu_position', $tmcptg_meta ) && $tmcptg_meta['tmcptg_menu_position'][0] ? (int) $tmcptg_meta['tmcptg_menu_position'][0] : null );
				$tmcptg_public              = ( array_key_exists( 'tmcptg_public', $tmcptg_meta ) && $tmcptg_meta['tmcptg_public'][0] == '1' ? true : false );
				$tmcptg_show_ui             = ( array_key_exists( 'tmcptg_show_ui', $tmcptg_meta ) && $tmcptg_meta['tmcptg_show_ui'][0] == '1' ? true : false );
				$tmcptg_has_archive         = ( array_key_exists( 'tmcptg_has_archive', $tmcptg_meta ) && $tmcptg_meta['tmcptg_has_archive'][0] == '1' ? true : false );
				$tmcptg_exclude_from_search = ( array_key_exists( 'tmcptg_exclude_from_search', $tmcptg_meta ) && $tmcptg_meta['tmcptg_exclude_from_search'][0] == '1' ? true : false );
				$tmcptg_capability_type     = ( array_key_exists( 'tmcptg_capability_type', $tmcptg_meta ) && $tmcptg_meta['tmcptg_capability_type'][0] ? $tmcptg_meta['tmcptg_capability_type'][0] : 'post' );
				$tmcptg_hierarchical        = ( array_key_exists( 'tmcptg_hierarchical', $tmcptg_meta ) && $tmcptg_meta['tmcptg_hierarchical'][0] == '1' ? true : false );
				$tmcptg_rewrite             = ( array_key_exists( 'tmcptg_rewrite', $tmcptg_meta ) && $tmcptg_meta['tmcptg_rewrite'][0] == '1' ? true : false );
				$tmcptg_withfront           = ( array_key_exists( 'tmcptg_withfront', $tmcptg_meta ) && $tmcptg_meta['tmcptg_withfront'][0] == '1' ? true : false );
				$tmcptg_feeds               = ( array_key_exists( 'tmcptg_feeds', $tmcptg_meta ) && $tmcptg_meta['tmcptg_feeds'][0] == '1' ? true : false );
				$tmcptg_pages               = ( array_key_exists( 'tmcptg_pages', $tmcptg_meta ) && $tmcptg_meta['tmcptg_pages'][0] == '1' ? true : false );
				$tmcptg_query_var           = ( array_key_exists( 'tmcptg_query_var', $tmcptg_meta ) && $tmcptg_meta['tmcptg_query_var'][0] == '1' ? true : false );
				
				if ( ! array_key_exists( 'tmcptg_publicly_queryable', $tmcptg_meta ) ) {
					$tmcptg_publicly_queryable = true;
				} elseif ( $tmcptg_meta['tmcptg_publicly_queryable'][0] == '1' ) {
					$tmcptg_publicly_queryable = true;
				} else {
					$tmcptg_publicly_queryable = false;
				}
				$tmcptg_show_in_menu        = ( array_key_exists( 'tmcptg_show_in_menu', $tmcptg_meta ) && $tmcptg_meta['tmcptg_show_in_menu'][0] == '1' ? true : false );
				$tmcptg_supports            = ( array_key_exists( 'tmcptg_supports', $tmcptg_meta ) && $tmcptg_meta['tmcptg_supports'][0] ? $tmcptg_meta['tmcptg_supports'][0] : 'a:2:{i:0;s:5:"title";i:1;s:6:"editor";}' );
				$tmcptg_builtin_taxonomies  = ( array_key_exists( 'tmcptg_builtin_taxonomies', $tmcptg_meta ) && $tmcptg_meta['tmcptg_builtin_taxonomies'][0] ? $tmcptg_meta['tmcptg_builtin_taxonomies'][0] : 'a:0:{}' );
				$tmcptg_rewrite_options     = array();
				if ( $tmcptg_rewrite )      { $tmcptg_rewrite_options['slug'] = _x( $tmcptg_custom_rewrite_slug, 'URL Slug', 'ten-minutes-custom-post-type-generator' ); }
				
				$tmcptg_rewrite_options['with_front'] = $tmcptg_withfront;
				if ( $tmcptg_feeds )        { $tmcptg_rewrite_options['feeds'] = $tmcptg_feeds; }
				if ( $tmcptg_pages )        { $tmcptg_rewrite_options['pages'] = $tmcptg_pages; }
				$tmcptgs[] = array(
					'tmcptg_id'                  => $tmcptg->ID,
					'tmcptg_name'                => $tmcptg_name,
					'tmcptg_label'               => $tmcptg_label,
					'tmcptg_singular_name'       => $tmcptg_singular_name,
					'tmcptg_description'         => $tmcptg_description,
					'tmcptg_icon_name'           => $tmcptg_icon_name,
					'tmcptg_custom_rewrite_slug' => $tmcptg_custom_rewrite_slug,
					'tmcptg_menu_position'       => $tmcptg_menu_position,
					'tmcptg_public'              => (bool) $tmcptg_public,
					'tmcptg_show_ui'             => (bool) $tmcptg_show_ui,
					'tmcptg_has_archive'         => (bool) $tmcptg_has_archive,
					'tmcptg_exclude_from_search' => (bool) $tmcptg_exclude_from_search,
					'tmcptg_capability_type'     => $tmcptg_capability_type,
					'tmcptg_hierarchical'        => (bool) $tmcptg_hierarchical,
					'tmcptg_rewrite'             => $tmcptg_rewrite_options,
					'tmcptg_query_var'           => (bool) $tmcptg_query_var,
					'tmcptg_publicly_queryable'  => (bool) $tmcptg_publicly_queryable,
					'tmcptg_show_in_menu'        => (bool) $tmcptg_show_in_menu,
					'tmcptg_supports'            => unserialize( $tmcptg_supports ),
					'tmcptg_builtin_taxonomies'  => unserialize( $tmcptg_builtin_taxonomies ),
				);
				
				if ( is_array( $tmcptgs ) ) {
					foreach ($tmcptgs as $tmcptg_post_type) {
						$labels = array(
							'name'                => __( $tmcptg_post_type['tmcptg_label'], 'ten-minutes-custom-post-type-generator' ),
							'singular_name'       => __( $tmcptg_post_type['tmcptg_singular_name'], 'ten-minutes-custom-post-type-generator' ),
							'add_new'             => __( 'Add New' , 'ten-minutes-custom-post-type-generator' ),
							'add_new_item'        => __( 'Add New ' . $tmcptg_post_type['tmcptg_singular_name'] , 'ten-minutes-custom-post-type-generator' ),
							'edit_item'           => __( 'Edit ' . $tmcptg_post_type['tmcptg_singular_name'] , 'ten-minutes-custom-post-type-generator' ),
							'new_item'            => __( 'New ' . $tmcptg_post_type['tmcptg_singular_name'] , 'ten-minutes-custom-post-type-generator' ),
							'view_item'           => __( 'View ' . $tmcptg_post_type['tmcptg_singular_name'], 'ten-minutes-custom-post-type-generator' ),
							'search_items'        => __( 'Search ' . $tmcptg_post_type['tmcptg_label'], 'ten-minutes-custom-post-type-generator' ),
							'not_found'           => __( 'No ' .  $tmcptg_post_type['tmcptg_label'] . ' found', 'ten-minutes-custom-post-type-generator' ),
							'not_found_in_trash'  => __( 'No ' .  $tmcptg_post_type['tmcptg_label'] . ' found in Trash', 'ten-minutes-custom-post-type-generator' ),
						);
						$args = array(
							'labels'              => $labels,
							'description'         => $tmcptg_post_type['tmcptg_description'],
							'menu_icon'           => $tmcptg_post_type['tmcptg_icon_name'],
							'rewrite'             => $tmcptg_post_type['tmcptg_rewrite'],
							'menu_position'       => $tmcptg_post_type['tmcptg_menu_position'],
							'public'              => $tmcptg_post_type['tmcptg_public'],
							'show_ui'             => $tmcptg_post_type['tmcptg_show_ui'],
							'has_archive'         => $tmcptg_post_type['tmcptg_has_archive'],
							'exclude_from_search' => $tmcptg_post_type['tmcptg_exclude_from_search'],
							'capability_type'     => $tmcptg_post_type['tmcptg_capability_type'],
							'hierarchical'        => $tmcptg_post_type['tmcptg_hierarchical'],
							'show_in_menu'        => $tmcptg_post_type['tmcptg_show_in_menu'],
							'query_var'           => $tmcptg_post_type['tmcptg_query_var'],
							'publicly_queryable'  => $tmcptg_post_type['tmcptg_publicly_queryable'],
							'_builtin'            => false,
							'supports'            => $tmcptg_post_type['tmcptg_supports'],
							'taxonomies'          => $tmcptg_post_type['tmcptg_builtin_taxonomies']
						);
						if ( $tmcptg_post_type['tmcptg_name'] != 'no_name' )
							register_post_type( $tmcptg_post_type['tmcptg_name'], $args);
					}
				}
			}
		}
		$get_tmcptg_tax = array(
			'numberposts' 	   => -1,
			'post_type' 	   => 'tmcptg_tax',
			'post_status'      => 'publish',
			'suppress_filters' => false,
		);
		$tmcptg_taxonomies = get_posts( $get_tmcptg_tax );
		if ( $tmcptg_taxonomies ) {
			foreach ( $tmcptg_taxonomies as $tmcptg_tax ) {
				$tmcptg_meta = get_post_meta( $tmcptg_tax->ID, '', true );
				$tmcptg_tax_name                = ( array_key_exists( 'tmcptg_tax_name', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_name'][0] ? esc_html( $tmcptg_meta['tmcptg_tax_name'][0] ) : 'no_name' );
				$tmcptg_tax_label               = ( array_key_exists( 'tmcptg_tax_label', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_label'][0] ? esc_html( $tmcptg_meta['tmcptg_tax_label'][0] ) : $tmcptg_tax_name );
				$tmcptg_tax_singular_name       = ( array_key_exists( 'tmcptg_tax_singular_name', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_singular_name'][0] ? esc_html( $tmcptg_meta['tmcptg_tax_singular_name'][0] ) : $tmcptg_tax_label );
				$tmcptg_tax_custom_rewrite_slug = ( array_key_exists( 'tmcptg_tax_custom_rewrite_slug', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_custom_rewrite_slug'][0] ? esc_html( $tmcptg_meta['tmcptg_tax_custom_rewrite_slug'][0] ) : $tmcptg_tax_name );
				$tmcptg_tax_show_ui             = ( array_key_exists( 'tmcptg_tax_show_ui', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_show_ui'][0] == '1' ? true : false );
				$tmcptg_tax_hierarchical        = ( array_key_exists( 'tmcptg_tax_hierarchical', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_hierarchical'][0] == '1' ? true : false );
				$tmcptg_tax_rewrite             = ( array_key_exists( 'tmcptg_tax_rewrite', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_rewrite'][0] == '1' ? array( 'slug' => _x( $tmcptg_tax_custom_rewrite_slug, 'URL Slug', 'ten-minutes-custom-post-type-generator' ) ) : false );
				$tmcptg_tax_query_var           = ( array_key_exists( 'tmcptg_tax_query_var', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_query_var'][0] == '1' ? true : false );
				$tmcptg_tax_post_types          = ( array_key_exists( 'tmcptg_tax_post_types', $tmcptg_meta ) && $tmcptg_meta['tmcptg_tax_post_types'][0] ? $tmcptg_meta['tmcptg_tax_post_types'][0] : 'a:0:{}' );
				$tmcptg_taxs[] = array(
					'tmcptg_tax_id'                  => $tmcptg_tax->ID,
					'tmcptg_tax_name'                => $tmcptg_tax_name,
					'tmcptg_tax_label'               => $tmcptg_tax_label,
					'tmcptg_tax_singular_name'       => $tmcptg_tax_singular_name,
					'tmcptg_tax_custom_rewrite_slug' => $tmcptg_tax_custom_rewrite_slug,
					'tmcptg_tax_show_ui'             => (bool) $tmcptg_tax_show_ui,
					'tmcptg_tax_hierarchical'        => (bool) $tmcptg_tax_hierarchical,
					'tmcptg_tax_rewrite'             => $tmcptg_tax_rewrite,
					'tmcptg_tax_query_var'           => (bool) $tmcptg_tax_query_var,
					'tmcptg_tax_builtin_taxonomies'  => unserialize( $tmcptg_tax_post_types ),
				);
				if ( is_array( $tmcptg_taxs ) ) {
					foreach ($tmcptg_taxs as $tmcptg_taxonomy) {
						$labels = array(
							'name'                       => _x( $tmcptg_taxonomy['tmcptg_tax_label'], 'taxonomy general name', 'ten-minutes-custom-post-type-generator' ),
							'singular_name'              => _x( $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'taxonomy singular name' ),
							'search_items'               => __( 'Search ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' ),
							'popular_items'              => __( 'Popular ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' ),
							'all_items'                  => __( 'All ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' ),
							'parent_item'                => __( 'Parent ' . $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'ten-minutes-custom-post-type-generator' ),
							'parent_item_colon'          => __( 'Parent ' . $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'ten-minutes-custom-post-type-generator' . ':' ),
							'edit_item'                  => __( 'Edit ' . $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'ten-minutes-custom-post-type-generator' ),
							'update_item'                => __( 'Update ' . $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'ten-minutes-custom-post-type-generator' ),
							'add_new_item'               => __( 'Add New ' . $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'ten-minutes-custom-post-type-generator' ),
							'new_item_name'              => __( 'New ' . $tmcptg_taxonomy['tmcptg_tax_singular_name'], 'ten-minutes-custom-post-type-generator' . ' Name' ),
							'separate_items_with_commas' => __( 'Seperate ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' . ' with commas' ),
							'add_or_remove_items'        => __( 'Add or remove ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' ),
							'choose_from_most_used'      => __( 'Choose from the most used ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' ),
							'menu_name'                  => __( 'All ' . $tmcptg_taxonomy['tmcptg_tax_label'], 'ten-minutes-custom-post-type-generator' )
						);
						$args = array(
							'label'               => $tmcptg_taxonomy['tmcptg_tax_label'],
							'labels'              => $labels,
							'rewrite'             => $tmcptg_taxonomy['tmcptg_tax_rewrite'],
							'show_ui'             => $tmcptg_taxonomy['tmcptg_tax_show_ui'],
							'hierarchical'        => $tmcptg_taxonomy['tmcptg_tax_hierarchical'],
							'query_var'           => $tmcptg_taxonomy['tmcptg_tax_query_var'],
						);
						if ( $tmcptg_taxonomy['tmcptg_tax_name'] != 'no_name' )
							register_taxonomy( $tmcptg_taxonomy['tmcptg_tax_name'], $tmcptg_taxonomy['tmcptg_tax_builtin_taxonomies'], $args );
					}
				}
			}
		}
	}
	public function tmcptg_create_meta_boxes()
	{
		add_meta_box(
			'tmcptg_options',
			__( 'Options', 'ten-minutes-custom-post-type-generator' ),
			array($this, 'tmcptg_meta_box'),
			'tmcptg',
			'normal',
			'high'
		);
		add_meta_box(
			'tmcptg_tax_options',
			__( 'Options', 'ten-minutes-custom-post-type-generator' ),
			array($this, 'tmcptg_tax_meta_box'),
			'tmcptg_tax',
			'normal',
			'high'
		);
	}
	
	public function tmcptg_meta_box( $post ) {
		$values = get_post_custom( $post->ID );
		$tmcptg_name                          = isset( $values['tmcptg_name'] ) ? esc_attr( $values['tmcptg_name'][0] ) : '';
		$tmcptg_label                         = isset( $values['tmcptg_label'] ) ? esc_attr( $values['tmcptg_label'][0] ) : '';
		$tmcptg_singular_name                 = isset( $values['tmcptg_singular_name'] ) ? esc_attr( $values['tmcptg_singular_name'][0] ) : '';
		$tmcptg_description                   = isset( $values['tmcptg_description'] ) ? esc_attr( $values['tmcptg_description'][0] ) : '';
		$tmcptg_icon_url                      = isset( $values['tmcptg_icon_url'] ) ? esc_attr( $values['tmcptg_icon_url'][0] ) : '';
		
		$tmcptg_icon_slug                     = isset( $values['tmcptg_icon_slug'] ) ? esc_attr( $values['tmcptg_icon_slug'][0] ) : ''; 
		
		if ( !empty($tmcptg_icon_slug) ) {
			$tmcptg_icon_name = $tmcptg_icon_slug;
		} else {
			$tmcptg_icon_name = $tmcptg_icon_url;
		}
		
		
		$tmcptg_custom_rewrite_slug           = isset( $values['tmcptg_custom_rewrite_slug'] ) ? esc_attr( $values['tmcptg_custom_rewrite_slug'][0] ) : '';
		$tmcptg_menu_position                 = isset( $values['tmcptg_menu_position'] ) ? esc_attr( $values['tmcptg_menu_position'][0] ) : '';
		$tmcptg_public                        = isset( $values['tmcptg_public'] ) ? esc_attr( $values['tmcptg_public'][0] ) : '';
		$tmcptg_show_ui                       = isset( $values['tmcptg_show_ui'] ) ? esc_attr( $values['tmcptg_show_ui'][0] ) : '';
		$tmcptg_has_archive                   = isset( $values['tmcptg_has_archive'] ) ? esc_attr( $values['tmcptg_has_archive'][0] ) : '';
		$tmcptg_exclude_from_search           = isset( $values['tmcptg_exclude_from_search'] ) ? esc_attr( $values['tmcptg_exclude_from_search'][0] ) : '';
		$tmcptg_capability_type               = isset( $values['tmcptg_capability_type'] ) ? esc_attr( $values['tmcptg_capability_type'][0] ) : '';
		$tmcptg_hierarchical                  = isset( $values['tmcptg_hierarchical'] ) ? esc_attr( $values['tmcptg_hierarchical'][0] ) : '';
		$tmcptg_rewrite                       = isset( $values['tmcptg_rewrite'] ) ? esc_attr( $values['tmcptg_rewrite'][0] ) : '';
		$tmcptg_withfront                     = isset( $values['tmcptg_withfront'] ) ? esc_attr( $values['tmcptg_withfront'][0] ) : '';
		$tmcptg_feeds                         = isset( $values['tmcptg_feeds'] ) ? esc_attr( $values['tmcptg_feeds'][0] ) : '';
		$tmcptg_pages                         = isset( $values['tmcptg_pages'] ) ? esc_attr( $values['tmcptg_pages'][0] ) : '';
		$tmcptg_query_var                     = isset( $values['tmcptg_query_var'] ) ? esc_attr( $values['tmcptg_query_var'][0] ) : '';
		$tmcptg_publicly_queryable            = isset( $values['tmcptg_publicly_queryable'] ) ? esc_attr( $values['tmcptg_publicly_queryable'][0] ) : '';
		$tmcptg_show_in_menu                  = isset( $values['tmcptg_show_in_menu'] ) ? esc_attr( $values['tmcptg_show_in_menu'][0] ) : '';
		$tmcptg_supports                      = isset( $values['tmcptg_supports'] ) ? unserialize( $values['tmcptg_supports'][0] ) : array();
		$tmcptg_supports_title                = ( isset( $values['tmcptg_supports'] ) && in_array( 'title', $tmcptg_supports ) ? 'title' : '' );
		$tmcptg_supports_editor               = ( isset( $values['tmcptg_supports'] ) && in_array( 'editor', $tmcptg_supports ) ? 'editor' : '' );
		$tmcptg_supports_excerpt              = ( isset( $values['tmcptg_supports'] ) && in_array( 'excerpt', $tmcptg_supports ) ? 'excerpt' : '' );
		$tmcptg_supports_trackbacks           = ( isset( $values['tmcptg_supports'] ) && in_array( 'trackbacks', $tmcptg_supports ) ? 'trackbacks' : '' );
		$tmcptg_supports_custom_fields        = ( isset( $values['tmcptg_supports'] ) && in_array( 'custom-fields', $tmcptg_supports ) ? 'custom-fields' : '' );
		$tmcptg_supports_comments             = ( isset( $values['tmcptg_supports'] ) && in_array( 'comments', $tmcptg_supports ) ? 'comments' : '' );
		$tmcptg_supports_revisions            = ( isset( $values['tmcptg_supports'] ) && in_array( 'revisions', $tmcptg_supports ) ? 'revisions' : '' );
		$tmcptg_supports_featured_image       = ( isset( $values['tmcptg_supports'] ) && in_array( 'thumbnail', $tmcptg_supports ) ? 'thumbnail' : '' );
		$tmcptg_supports_author               = ( isset( $values['tmcptg_supports'] ) && in_array( 'author', $tmcptg_supports ) ? 'author' : '' );
		$tmcptg_supports_page_attributes      = ( isset( $values['tmcptg_supports'] ) && in_array( 'page-attributes', $tmcptg_supports ) ? 'page-attributes' : '' );
		$tmcptg_supports_post_formats         = ( isset( $values['tmcptg_supports'] ) && in_array( 'post-formats', $tmcptg_supports ) ? 'post-formats' : '' );
		$tmcptg_builtin_taxonomies            = isset( $values['tmcptg_builtin_taxonomies'] ) ? unserialize( $values['tmcptg_builtin_taxonomies'][0] ) : array();
		$tmcptg_builtin_taxonomies_categories = ( isset( $values['tmcptg_builtin_taxonomies'] ) && in_array( 'category', $tmcptg_builtin_taxonomies ) ? 'category' : '' );
		$tmcptg_builtin_taxonomies_tags       = ( isset( $values['tmcptg_builtin_taxonomies'] ) && in_array( 'post_tag', $tmcptg_builtin_taxonomies ) ? 'post_tag' : '' );
		wp_nonce_field( 'tmcptg_meta_box_nonce_action', 'tmcptg_meta_box_nonce_field' );
		global $pagenow;
		$tmcptg_supports_title                = $pagenow === 'post-new.php' ? 'title' : $tmcptg_supports_title;
		$tmcptg_supports_editor               = $pagenow === 'post-new.php' ? 'editor' : $tmcptg_supports_editor;
		$tmcptg_supports_excerpt              = $pagenow === 'post-new.php' ? 'excerpt' : $tmcptg_supports_excerpt;
		?>
    <table class="tmcptg">
        <tr>
            <td class="label">
                <label for="tmcptg_name"><span class="required">*</span> <?php _e( 'Custom Post Type Name', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="text" name="tmcptg_name" id="tmcptg_name" class="widefat" tabindex="1" value="<?php echo $tmcptg_name; ?>" />
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_label"><?php _e( 'Label', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="text" name="tmcptg_label" id="tmcptg_label" class="widefat" tabindex="2" value="<?php echo $tmcptg_label; ?>" />
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_singular_name"><?php _e( 'Singular Name', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="text" name="tmcptg_singular_name" id="tmcptg_singular_name" class="widefat" tabindex="3" value="<?php echo $tmcptg_singular_name; ?>" />
            </td>
        </tr>
        <tr>
            <td class="label top">
                <label for="tmcptg_description"><?php _e( 'Description', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <textarea name="tmcptg_description" id="tmcptg_description" class="widefat" tabindex="4" rows="4"><?php echo $tmcptg_description; ?></textarea>
            </td>
        </tr>
        <tr>
            <td colspan="2" class="section">
                <h3>
                    <?php _e( 'Visibility', 'ten-minutes-custom-post-type-generator' ); ?>
                </h3>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_public"><?php _e( 'Public', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_public" id="tmcptg_public" tabindex="5">
						<option value="1" <?php selected( $tmcptg_public, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_public, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td colspan="2" class="section">
                <h3>
                    <?php _e( 'Rewrite Options', 'ten-minutes-custom-post-type-generator' ); ?>
                </h3>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_rewrite"><?php _e( 'Rewrite', 'ten-minutes-custom-post-type-generator' ); ?></label
				</td>
				<td>
					<select name="tmcptg_rewrite" id="tmcptg_rewrite" tabindex="6">
						<option value="1" <?php selected( $tmcptg_rewrite, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_rewrite, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
				</td>
			</tr>
			<tr>
				<td class="label">
					<label for="tmcptg_withfront"><?php _e( 'With Front', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_withfront" id="tmcptg_withfront" tabindex="7">
						<option value="1" <?php selected( $tmcptg_withfront, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_withfront, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_custom_rewrite_slug"><?php _e( 'Custom Rewrite Slug', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="text" name="tmcptg_custom_rewrite_slug" id="tmcptg_custom_rewrite_slug" class="widefat" tabindex="8" value="<?php echo $tmcptg_custom_rewrite_slug; ?>" />
            </td>
        </tr>
        <tr>
            <td colspan="2" class="section">
                <h3>
                    <?php _e( 'Front-end Options', 'ten-minutes-custom-post-type-generator' ); ?>
                </h3>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_feeds"><?php _e( 'Feeds', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_feeds" id="tmcptg_feeds" tabindex="9">
						<option value="0" <?php selected( $tmcptg_feeds, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="1" <?php selected( $tmcptg_feeds, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_pages"><?php _e( 'Pages', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_pages" id="tmcptg_pages" tabindex="10">
						<option value="1" <?php selected( $tmcptg_pages, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_pages, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_exclude_from_search"><?php _e( 'Exclude From Search', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_exclude_from_search" id="tmcptg_exclude_from_search" tabindex="11">
						<option value="0" <?php selected( $tmcptg_exclude_from_search, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="1" <?php selected( $tmcptg_exclude_from_search, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_has_archive"><?php _e( 'Has Archive', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_has_archive" id="tmcptg_has_archive" tabindex="12">
						<option value="0" <?php selected( $tmcptg_has_archive, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="1" <?php selected( $tmcptg_has_archive, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td colspan="2" class="section">
                <h3>
                    <?php _e( 'Admin Menu Options', 'ten-minutes-custom-post-type-generator' ); ?>
                </h3>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_show_ui"><?php _e( 'Show UI', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_show_ui" id="tmcptg_show_ui" tabindex="13">
						<option value="1" <?php selected( $tmcptg_show_ui, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_show_ui, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_menu_position"><?php _e( 'Menu Position', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="text" name="tmcptg_menu_position" id="tmcptg_menu_position" class="widefat" tabindex="14" value="<?php echo $tmcptg_menu_position; ?>" />
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_show_in_menu"><?php _e( 'Show in Menu', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_show_in_menu" id="tmcptg_show_in_menu" tabindex="15">
						<option value="1" <?php selected( $tmcptg_show_in_menu, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_show_in_menu, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="current-tmcptg-icon"><?php _e( 'Icon Image', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <div class="tmcptg-icon">
                    <div class="current-tmcptg-icon">
                        <?php if ( $tmcptg_icon_url ) { ?><img src="<?php echo $tmcptg_icon_url; ?>" />
                        <?php } ?>
                    </div>
                    <a href="/" class="remove-tmcptg-icon button-secondary" <?php if ( ! $tmcptg_icon_url ) { ?> style="display: none;"<?php } ?>>Remove icon</a>
                    <a href="/" class="media-uploader-button button-primary" data-post-id="<?php echo $post->ID; ?>">
                        <?php if ( ! $tmcptg_icon_url ) { ?>
                        <?php _e( 'Add icon image', 'ten-minutes-custom-post-type-generator' ); ?>
                        <?php } else { ?>
                        <?php _e( 'Edit icon', 'ten-minutes-custom-post-type-generator' ); ?>
                        <?php } ?>
                    </a>
                </div>
                <input type="hidden" name="tmcptg_icon_url" id="tmcptg_icon_url" class="widefat" value="<?php echo $tmcptg_icon_url; ?>" />
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_icon_slug"><?php _e( 'Slug Icon', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <?php if ( $tmcptg_icon_slug ) { ?><span id="tmcptg_icon_slug_before" class="dashicons-before <?php echo $tmcptg_icon_slug; ?>"><?php } ?></span>
                <input type="text" name="tmcptg_icon_slug" id="tmcptg_icon_slug" class="widefat" tabindex="15" value="<?php echo $tmcptg_icon_slug; ?>" />
            </td>
        </tr>
        <tr>
            <td colspan="2" class="section">
                <h3>
                    <?php _e( 'Wordpress Integration', 'ten-minutes-custom-post-type-generator' ); ?>
                </h3>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_capability_type"><?php _e( 'Capability Type', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_capability_type" id="tmcptg_capability_type" tabindex="16">
						<option value="post" <?php selected( $tmcptg_capability_type, 'post' ); ?>><?php _e( 'Post', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="page" <?php selected( $tmcptg_capability_type, 'page' ); ?>><?php _e( 'Page', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_hierarchical"><?php _e( 'Hierarchical', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_hierarchical" id="tmcptg_hierarchical" tabindex="17">
						<option value="0" <?php selected( $tmcptg_hierarchical, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="1" <?php selected( $tmcptg_hierarchical, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_query_var"><?php _e( 'Query Var', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_query_var" id="tmcptg_query_var" tabindex="18">
						<option value="1" <?php selected( $tmcptg_query_var, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_query_var, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label">
                <label for="tmcptg_publicly_queryable"><?php _e( 'Publicly Queryable', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <select name="tmcptg_publicly_queryable" id="tmcptg_publicly_queryable" tabindex="18">
						<option value="1" <?php selected( $tmcptg_publicly_queryable, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_publicly_queryable, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
            </td>
        </tr>
        <tr>
            <td class="label top">
                <label for="tmcptg_supports"><?php _e( 'Supports', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="checkbox" tabindex="19" name="tmcptg_supports[]" id="tmcptg_supports_title" value="title" <?php checked( $tmcptg_supports_title, 'title' ); ?> /> <label for="tmcptg_supports_title"><?php _e( 'Title', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="20" name="tmcptg_supports[]" id="tmcptg_supports_editor" value="editor" <?php checked( $tmcptg_supports_editor, 'editor' ); ?> /> <label for="tmcptg_supports_editor"><?php _e( 'Editor', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="21" name="tmcptg_supports[]" id="tmcptg_supports_excerpt" value="excerpt" <?php checked( $tmcptg_supports_excerpt, 'excerpt' ); ?> /> <label for="tmcptg_supports_excerpt"><?php _e( 'Excerpt', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="22" name="tmcptg_supports[]" id="tmcptg_supports_trackbacks" value="trackbacks" <?php checked( $tmcptg_supports_trackbacks, 'trackbacks' ); ?> /> <label for="tmcptg_supports_trackbacks"><?php _e( 'Trackbacks', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="23" name="tmcptg_supports[]" id="tmcptg_supports_custom_fields" value="custom-fields" <?php checked( $tmcptg_supports_custom_fields, 'custom-fields' ); ?> /> <label for="tmcptg_supports_custom_fields"><?php _e( 'Custom Fields', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="24" name="tmcptg_supports[]" id="tmcptg_supports_comments" value="comments" <?php checked( $tmcptg_supports_comments, 'comments' ); ?> /> <label for="tmcptg_supports_comments"><?php _e( 'Comments', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="25" name="tmcptg_supports[]" id="tmcptg_supports_revisions" value="revisions" <?php checked( $tmcptg_supports_revisions, 'revisions' ); ?> /> <label for="tmcptg_supports_revisions"><?php _e( 'Revisions', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="26" name="tmcptg_supports[]" id="tmcptg_supports_featured_image" value="thumbnail" <?php checked( $tmcptg_supports_featured_image, 'thumbnail' ); ?> /> <label for="tmcptg_supports_featured_image"><?php _e( 'Featured Image', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="27" name="tmcptg_supports[]" id="tmcptg_supports_author" value="author" <?php checked( $tmcptg_supports_author, 'author' ); ?> /> <label for="tmcptg_supports_author"><?php _e( 'Author', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="28" name="tmcptg_supports[]" id="tmcptg_supports_page_attributes" value="page-attributes" <?php checked( $tmcptg_supports_page_attributes, 'page-attributes' ); ?> /> <label for="tmcptg_supports_page_attributes"><?php _e( 'Page Attributes', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="29" name="tmcptg_supports[]" id="tmcptg_supports_post_formats" value="post-formats" <?php checked( $tmcptg_supports_post_formats, 'post-formats' ); ?> /> <label for="tmcptg_supports_post_formats"><?php _e( 'Post Formats', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
            </td>
        </tr>
        <tr>
            <td class="label top">
                <label for="tmcptg_builtin_taxonomies"><?php _e( 'Built-in Taxonomies', 'ten-minutes-custom-post-type-generator' ); ?></label>
            </td>
            <td>
                <input type="checkbox" tabindex="30" name="tmcptg_builtin_taxonomies[]" id="tmcptg_builtin_taxonomies_categories" value="category" <?php checked( $tmcptg_builtin_taxonomies_categories, 'category' ); ?> /> <label for="tmcptg_builtin_taxonomies_categories"><?php _e( 'Categories', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                <input type="checkbox" tabindex="31" name="tmcptg_builtin_taxonomies[]" id="tmcptg_builtin_taxonomies_tags" value="post_tag" <?php checked( $tmcptg_builtin_taxonomies_tags, 'post_tag' ); ?> /> <label for="tmcptg_builtin_taxonomies_tags"><?php _e( 'Tags', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
            </td>
        </tr>
    </table>
    <?php
	}
	public function tmcptg_tax_meta_box( $post ) 
	{
		$values = get_post_custom( $post->ID );
		$tmcptg_tax_name                          = isset( $values['tmcptg_tax_name'] ) ? esc_attr( $values['tmcptg_tax_name'][0] ) : '';
		$tmcptg_tax_label                         = isset( $values['tmcptg_tax_label'] ) ? esc_attr( $values['tmcptg_tax_label'][0] ) : '';
		$tmcptg_tax_singular_name                 = isset( $values['tmcptg_tax_singular_name'] ) ? esc_attr( $values['tmcptg_tax_singular_name'][0] ) : '';
		$tmcptg_tax_custom_rewrite_slug           = isset( $values['tmcptg_tax_custom_rewrite_slug'] ) ? esc_attr( $values['tmcptg_tax_custom_rewrite_slug'][0] ) : '';
		$tmcptg_tax_show_ui                       = isset( $values['tmcptg_tax_show_ui'] ) ? esc_attr( $values['tmcptg_tax_show_ui'][0] ) : '';
		$tmcptg_tax_hierarchical                  = isset( $values['tmcptg_tax_hierarchical'] ) ? esc_attr( $values['tmcptg_tax_hierarchical'][0] ) : '';
		$tmcptg_tax_rewrite                       = isset( $values['tmcptg_tax_rewrite'] ) ? esc_attr( $values['tmcptg_tax_rewrite'][0] ) : '';
		$tmcptg_tax_query_var                     = isset( $values['tmcptg_tax_query_var'] ) ? esc_attr( $values['tmcptg_tax_query_var'][0] ) : '';
		$tmcptg_tax_supports                      = isset( $values['tmcptg_tax_supports'] ) ? unserialize( $values['tmcptg_tax_supports'][0] ) : array();
		$tmcptg_tax_supports_title                = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'title', $tmcptg_supports ) ? 'title' : '' );
		$tmcptg_tax_supports_editor               = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'editor', $tmcptg_supports ) ? 'editor' : '' );
		$tmcptg_tax_supports_excerpt              = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'excerpt', $tmcptg_supports ) ? 'excerpt' : '' );
		$tmcptg_tax_supports_trackbacks           = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'trackbacks', $tmcptg_supports ) ? 'trackbacks' : '' );
		$tmcptg_tax_supports_custom_fields        = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'custom-fields', $tmcptg_supports ) ? 'custom-fields' : '' );
		$tmcptg_tax_supports_comments             = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'comments', $tmcptg_supports ) ? 'comments' : '' );
		$tmcptg_tax_supports_revisions            = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'revisions', $tmcptg_supports ) ? 'revisions' : '' );
		$tmcptg_tax_supports_featured_image       = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'thumbnail', $tmcptg_supports ) ? 'thumbnail' : '' );
		$tmcptg_tax_supports_author               = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'author', $tmcptg_supports ) ? 'author' : '' );
		$tmcptg_tax_supports_page_attributes      = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'page-attributes', $tmcptg_supports ) ? 'page-attributes' : '' );
		$tmcptg_tax_supports_post_formats         = ( isset( $values['tmcptg_tax_supports'] ) && in_array( 'post-formats', $tmcptg_supports ) ? 'post-formats' : '' );
		$tmcptg_tax_post_types                    = isset( $values['tmcptg_tax_post_types'] ) ? unserialize( $values['tmcptg_tax_post_types'][0] ) : array();
		$tmcptg_tax_post_types_post               = ( isset( $values['tmcptg_tax_post_types'] ) && in_array( 'post', $tmcptg_tax_post_types ) ? 'post' : '' );
		$tmcptg_tax_post_types_page               = ( isset( $values['tmcptg_tax_post_types'] ) && in_array( 'page', $tmcptg_tax_post_types ) ? 'page' : '' );
		wp_nonce_field( 'tmcptg_meta_box_nonce_action', 'tmcptg_meta_box_nonce_field' );
		?>
        <table class="tmcptg">
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_name"><span class="required">*</span> <?php _e( 'Custom Taxonomy Name', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <input type="text" name="tmcptg_tax_name" id="tmcptg_tax_name" class="widefat" tabindex="1" value="<?php echo $tmcptg_tax_name; ?>" />
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_label"><?php _e( 'Label', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <input type="text" name="tmcptg_tax_label" id="tmcptg_tax_label" class="widefat" tabindex="2" value="<?php echo $tmcptg_tax_label; ?>" />
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_singular_name"><?php _e( 'Singular Name', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <input type="text" name="tmcptg_tax_singular_name" id="tmcptg_tax_singular_name" class="widefat" tabindex="3" value="<?php echo $tmcptg_tax_singular_name; ?>" />
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_show_ui"><?php _e( 'Show UI', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <select name="tmcptg_tax_show_ui" id="tmcptg_tax_show_ui" tabindex="4">
						<option value="1" <?php selected( $tmcptg_tax_show_ui, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_tax_show_ui, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_hierarchical"><?php _e( 'Hierarchical', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <select name="tmcptg_tax_hierarchical" id="tmcptg_tax_hierarchical" tabindex="5">
						<option value="0" <?php selected( $tmcptg_tax_hierarchical, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="1" <?php selected( $tmcptg_tax_hierarchical, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_rewrite"><?php _e( 'Rewrite', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <select name="tmcptg_tax_rewrite" id="tmcptg_tax_rewrite" tabindex="6">
						<option value="1" <?php selected( $tmcptg_tax_rewrite, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_tax_rewrite, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_custom_rewrite_slug"><?php _e( 'Custom Rewrite Slug', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <input type="text" name="tmcptg_tax_custom_rewrite_slug" id="tmcptg_tax_custom_rewrite_slug" class="widefat" tabindex="7" value="<?php echo $tmcptg_tax_custom_rewrite_slug; ?>" />
                </td>
            </tr>
            <tr>
                <td class="label">
                    <label for="tmcptg_tax_query_var"><?php _e( 'Query Var', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <select name="tmcptg_tax_query_var" id="tmcptg_tax_query_var" tabindex="8">
						<option value="1" <?php selected( $tmcptg_tax_query_var, '1' ); ?>><?php _e( 'True', 'ten-minutes-custom-post-type-generator' ); ?></option>
						<option value="0" <?php selected( $tmcptg_tax_query_var, '0' ); ?>><?php _e( 'False', 'ten-minutes-custom-post-type-generator' ); ?></option>
					</select>
                </td>
            </tr>
            <tr>
                <td class="label top">
                    <label for="tmcptg_tax_post_types"><?php _e( 'Post Types', 'ten-minutes-custom-post-type-generator' ); ?></label>
                </td>
                <td>
                    <input type="checkbox" tabindex="9" name="tmcptg_tax_post_types[]" id="tmcptg_tax_post_types_post" value="post" <?php checked( $tmcptg_tax_post_types_post, 'post' ); ?> /> <label for="tmcptg_tax_post_types_post"><?php _e( 'Posts', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                    <input type="checkbox" tabindex="10" name="tmcptg_tax_post_types[]" id="tmcptg_tax_post_types_page" value="page" <?php checked( $tmcptg_tax_post_types_page, 'page' ); ?> /> <label for="tmcptg_tax_post_types_page"><?php _e( 'Pages', 'ten-minutes-custom-post-type-generator' ); ?></label><br />
                    <?php
						$post_types = get_post_types( array( 'public' => true, '_builtin' => false ) );
						$i = 10;
						foreach ( $post_types as $post_type ) {
							$checked = in_array( $post_type, $tmcptg_tax_post_types )  ? 'checked="checked"' : '';
							?>
                        <input type="checkbox" tabindex="<?php echo $i; ?>" name="tmcptg_tax_post_types[]" id="tmcptg_tax_post_types_<?php echo $post_type; ?>" value="<?php echo $post_type; ?>" <?php echo $checked; ?> /> <label for="tmcptg_tax_post_types_<?php echo $post_type; ?>"><?php echo ucfirst( $post_type ); ?></label><br />
                        <?php
							$i++;
						}
					?>
                </td>
            </tr>
        </table>
        <?php
	}
	
	public function tmcptg_save_post( $post_id ) 
	{
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )
			return;
		if ( !isset( $_POST['tmcptg_meta_box_nonce_field'] ) || !wp_verify_nonce( $_POST['tmcptg_meta_box_nonce_field'], 'tmcptg_meta_box_nonce_action' ) ) return;
		if ( isset($_POST['tmcptg_name']) )
			update_post_meta( $post_id, 'tmcptg_name', sanitize_text_field( str_replace( ' ', '', $_POST['tmcptg_name'] ) ) );
		if ( isset($_POST['tmcptg_label']) )
			update_post_meta( $post_id, 'tmcptg_label', sanitize_text_field( $_POST['tmcptg_label'] ) );
		if ( isset($_POST['tmcptg_singular_name']) )
			update_post_meta( $post_id, 'tmcptg_singular_name', sanitize_text_field( $_POST['tmcptg_singular_name'] ) );
		if ( isset($_POST['tmcptg_description']) )
			update_post_meta( $post_id, 'tmcptg_description', esc_textarea( $_POST['tmcptg_description'] ) );
		if ( isset($_POST['tmcptg_icon_slug']) )
			update_post_meta( $post_id, 'tmcptg_icon_slug', esc_textarea( $_POST['tmcptg_icon_slug'] ) );
		
        if ( isset($_POST['tmcptg_icon_url']) )
			update_post_meta( $post_id, 'tmcptg_icon_url', esc_textarea( $_POST['tmcptg_icon_url'] ) );
		if ( isset( $_POST['tmcptg_public'] ) )
			update_post_meta( $post_id, 'tmcptg_public', esc_attr( $_POST['tmcptg_public'] ) );
		if ( isset( $_POST['tmcptg_show_ui'] ) )
			update_post_meta( $post_id, 'tmcptg_show_ui', esc_attr( $_POST['tmcptg_show_ui'] ) );
		if ( isset( $_POST['tmcptg_has_archive'] ) )
			update_post_meta( $post_id, 'tmcptg_has_archive', esc_attr( $_POST['tmcptg_has_archive'] ) );
		if ( isset( $_POST['tmcptg_exclude_from_search'] ) )
			update_post_meta( $post_id, 'tmcptg_exclude_from_search', esc_attr( $_POST['tmcptg_exclude_from_search'] ) );
		if ( isset( $_POST['tmcptg_capability_type'] ) )
			update_post_meta( $post_id, 'tmcptg_capability_type', esc_attr( $_POST['tmcptg_capability_type'] ) );
		if ( isset( $_POST['tmcptg_hierarchical'] ) )
			update_post_meta( $post_id, 'tmcptg_hierarchical', esc_attr( $_POST['tmcptg_hierarchical'] ) );
		if ( isset( $_POST['tmcptg_rewrite'] ) )
			update_post_meta( $post_id, 'tmcptg_rewrite', esc_attr( $_POST['tmcptg_rewrite'] ) );
		if ( isset( $_POST['tmcptg_withfront'] ) )
			update_post_meta( $post_id, 'tmcptg_withfront', esc_attr( $_POST['tmcptg_withfront'] ) );
		if ( isset( $_POST['tmcptg_feeds'] ) )
			update_post_meta( $post_id, 'tmcptg_feeds', esc_attr( $_POST['tmcptg_feeds'] ) );
		if ( isset( $_POST['tmcptg_pages'] ) )
			update_post_meta( $post_id, 'tmcptg_pages', esc_attr( $_POST['tmcptg_pages'] ) );
		if ( isset($_POST['tmcptg_custom_rewrite_slug']) )
			update_post_meta( $post_id, 'tmcptg_custom_rewrite_slug', sanitize_text_field( $_POST['tmcptg_custom_rewrite_slug'] ) );
		if ( isset( $_POST['tmcptg_query_var'] ) )
			update_post_meta( $post_id, 'tmcptg_query_var', esc_attr( $_POST['tmcptg_query_var'] ) );
		
		if ( isset( $_POST['tmcptg_publicly_queryable'] ) )
			update_post_meta( $post_id, 'tmcptg_publicly_queryable', esc_attr( $_POST['tmcptg_publicly_queryable'] ) );
		if ( isset($_POST['tmcptg_menu_position']) )
			update_post_meta( $post_id, 'tmcptg_menu_position', sanitize_text_field( $_POST['tmcptg_menu_position'] ) );
		if ( isset( $_POST['tmcptg_show_in_menu'] ) )
			update_post_meta( $post_id, 'tmcptg_show_in_menu', esc_attr( $_POST['tmcptg_show_in_menu'] ) );
		$tmcptg_supports = isset( $_POST['tmcptg_supports'] ) ? $_POST['tmcptg_supports'] : array();
			update_post_meta( $post_id, 'tmcptg_supports', $tmcptg_supports );
		$tmcptg_builtin_taxonomies = isset( $_POST['tmcptg_builtin_taxonomies'] ) ? $_POST['tmcptg_builtin_taxonomies'] : array();
			update_post_meta( $post_id, 'tmcptg_builtin_taxonomies', $tmcptg_builtin_taxonomies );
		if ( isset($_POST['tmcptg_tax_name']) )
			update_post_meta( $post_id, 'tmcptg_tax_name', sanitize_text_field( str_replace( ' ', '', $_POST['tmcptg_tax_name'] ) ) );
		if ( isset($_POST['tmcptg_tax_label']) )
			update_post_meta( $post_id, 'tmcptg_tax_label', sanitize_text_field( $_POST['tmcptg_tax_label'] ) );
		if ( isset($_POST['tmcptg_tax_singular_name']) )
			update_post_meta( $post_id, 'tmcptg_tax_singular_name', sanitize_text_field( $_POST['tmcptg_tax_singular_name'] ) );
		if ( isset( $_POST['tmcptg_tax_show_ui'] ) )
			update_post_meta( $post_id, 'tmcptg_tax_show_ui', esc_attr( $_POST['tmcptg_tax_show_ui'] ) );
		if ( isset( $_POST['tmcptg_tax_hierarchical'] ) )
			update_post_meta( $post_id, 'tmcptg_tax_hierarchical', esc_attr( $_POST['tmcptg_tax_hierarchical'] ) );
		if ( isset( $_POST['tmcptg_tax_rewrite'] ) )
			update_post_meta( $post_id, 'tmcptg_tax_rewrite', esc_attr( $_POST['tmcptg_tax_rewrite'] ) );
		if ( isset($_POST['tmcptg_tax_custom_rewrite_slug']) )
			update_post_meta( $post_id, 'tmcptg_tax_custom_rewrite_slug', sanitize_text_field( $_POST['tmcptg_tax_custom_rewrite_slug'] ) );
		if ( isset( $_POST['tmcptg_tax_query_var'] ) )
			update_post_meta( $post_id, 'tmcptg_tax_query_var', esc_attr( $_POST['tmcptg_tax_query_var'] ) );
		$tmcptg_tax_post_types = isset( $_POST['tmcptg_tax_post_types'] ) ? $_POST['tmcptg_tax_post_types'] : array();
			update_post_meta( $post_id, 'tmcptg_tax_post_types', $tmcptg_tax_post_types );
			update_option( 'tmcptg_plugin_settings_changed', true );
	}
	
	function tmcptg_plugin_settings_flush_rewrite() 
	{
    if ( get_option( 'tmcptg_plugin_settings_changed' ) == true ) {
        flush_rewrite_rules();
        update_option( 'tmcptg_plugin_settings_changed', false );
    }
	}
	function tmcptg_plugin_activate_flush_rewrite() 
	{
		$this->tmcptg_create_custom_post_types();
		flush_rewrite_rules();
	}
	
	function tmcptg_change_columns( $cols ) 
	{
		$cols = array(
			'cb'                    => '<input type="checkbox" />',
			'title'                 => __( 'Post Type', 'ten-minutes-custom-post-type-generator' ),
			'custom_post_type_name' => __( 'Custom Post Type Name', 'ten-minutes-custom-post-type-generator' ),
			'label'                 => __( 'Label', 'ten-minutes-custom-post-type-generator' ),
			'description'           => __( 'Description', 'ten-minutes-custom-post-type-generator' ),
		);
		return $cols;
	}
	
	function tmcptg_sortable_columns() 
	{
		return array(
			'title'                 => 'title'
		);
	}
	function tmcptg_custom_columns( $column, $post_id ) 
	{
		switch ( $column ) {
			case "custom_post_type_name":
				echo get_post_meta( $post_id, 'tmcptg_name', true);
				break;
			case "label":
				echo get_post_meta( $post_id, 'tmcptg_label', true);
				break;
			case "description":
				echo get_post_meta( $post_id, 'tmcptg_description', true);
				break;
		}
	}
	
	function tmcptg_tax_change_columns( $cols ) 
	{
		$cols = array(
			'cb'                    => '<input type="checkbox" />',
			'title'                 => __( 'Taxonomy', 'ten-minutes-custom-post-type-generator' ),
			'custom_post_type_name' => __( 'Taxonomy Name', 'ten-minutes-custom-post-type-generator' ),
			'label'                 => __( 'Label', 'ten-minutes-custom-post-type-generator' )
		);
		return $cols;
	}
	
	function tmcptg_tax_sortable_columns() 
	{
		return array(
			'title'                 => 'title'
		);
	}
	
	function tmcptg_tax_custom_columns( $column, $post_id ) 
	{
		switch ( $column ) {
			case "custom_post_type_name":
				echo get_post_meta( $post_id, 'tmcptg_tax_name', true);
				break;
			case "label":
				echo get_post_meta( $post_id, 'tmcptg_tax_label', true);
				break;
		}
	}
	
	function tmcptg_post_updated_messages( $messages ) 
	{
		global $post, $post_ID;
		$messages['tmcptg' ] = array(
			0 => '', 
			1 => __( 'Custom Post Type updated.', 'ten-minutes-custom-post-type-generator' ),
			2 => __( 'Custom Post Type updated.', 'ten-minutes-custom-post-type-generator' ),
			3 => __( 'Custom Post Type deleted.', 'ten-minutes-custom-post-type-generator' ),
			4 => __( 'Custom Post Type updated.', 'ten-minutes-custom-post-type-generator' ),
			5 => isset($_GET['revision']) ? sprintf( __( 'Custom Post Type restored to revision from %s', 'ten-minutes-custom-post-type-generator' ), wp_post_revision_title( (int) $_GET['revision'], false ) ) : false,
			6 => __( 'Custom Post Type published.', 'ten-minutes-custom-post-type-generator' ),
			7 => __( 'Custom Post Type saved.', 'ten-minutes-custom-post-type-generator' ),
			8 => __( 'Custom Post Type submitted.', 'ten-minutes-custom-post-type-generator' ),
			9 => __( 'Custom Post Type scheduled for.', 'ten-minutes-custom-post-type-generator' ),
			10 => __( 'Custom Post Type draft updated.', 'ten-minutes-custom-post-type-generator' ),
		);
		return $messages;
	}
	
	function wp_prepare_attachment_for_js( $response, $attachment, $meta )
	{
		if ( $response['type'] != 'image' )
		{
			return $response;
		}
		$attachment_url = $response['url'];
		$base_url = str_replace( wp_basename( $attachment_url ), '', $attachment_url );
		if ( isset( $meta['sizes'] ) && is_array($meta['sizes']) )
		{
			foreach ( $meta['sizes'] as $k => $v )
			{
				if ( !isset($response['sizes'][ $k ]) )
				{
					$response['sizes'][ $k ] = array(
						'height'      =>  $v['height'],
						'width'       =>  $v['width'],
						'url'         => $base_url .  $v['file'],
						'orientation' => $v['height'] > $v['width'] ? 'portrait' : 'landscape',
					);
				}
			}
		}
		return $response;
	}
}
$tmcptg = new tmcptg();